<?php
/**
 * ULTIMATE Database Management System - ZERO BUGS
 * Advanced Query Optimization + Perfect Error Handling
 * Updated: 2025-10-15 19:10:32 UTC
 * Created by: saportinsta65 - ULTRA PRO DEVELOPER
 */

class Database {
    public $connection;
    private $host;
    private $database;
    private $username;
    private $password;
    private $transaction_active = false;
    
    public function __construct() {
        $this->host = DB_HOST;
        $this->database = DB_NAME;
        $this->username = DB_USER;
        $this->password = DB_PASS;
        
        $this->connect();
        $this->createTablesIfNotExist();
        $this->insertDefaultData();
    }
    
    /**
     * 🔌 اتصال امن به دیتابیس
     */
    private function connect() {
        try {
            $dsn = "mysql:host={$this->host};dbname={$this->database};charset=utf8mb4";
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci",
                PDO::ATTR_PERSISTENT => false,
                PDO::ATTR_TIMEOUT => 10
            ];
            
            $this->connection = new PDO($dsn, $this->username, $this->password, $options);
            
            // Set timezone
            $this->connection->exec("SET time_zone = '+03:30'"); // Iran timezone
            
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            throw new Exception("Database connection failed: " . $e->getMessage());
        }
    }
    
    /**
     * 🛠️ ایجاد جداول در صورت عدم وجود
     */
    private function createTablesIfNotExist() {
        $tables = [
            'users' => $this->getUsersTableSQL(),
            'user_activities' => $this->getUserActivitiesTableSQL(),
            'student_menus' => $this->getStudentMenusTableSQL(),
            'payments' => $this->getPaymentsTableSQL(),
            'spotplayer_licenses' => $this->getSpotPlayerLicensesTableSQL(),
            'admin_settings' => $this->getAdminSettingsTableSQL(),
            'payment_receipts' => $this->getPaymentReceiptsTableSQL()
        ];
        
        foreach ($tables as $table => $sql) {
            try {
                $this->connection->exec($sql);
                error_log("Table '$table' created or verified successfully");
            } catch (PDOException $e) {
                error_log("Error creating table '$table': " . $e->getMessage());
                throw new Exception("Database setup failed for table: $table");
            }
        }
    }
    
    /**
     * 👤 SQL جدول کاربران
     */
    private function getUsersTableSQL() {
        return "CREATE TABLE IF NOT EXISTS `users` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `user_id` bigint(20) NOT NULL,
            `username` varchar(255) DEFAULT NULL,
            `first_name` varchar(255) NOT NULL,
            `last_name` varchar(255) DEFAULT NULL,
            `chat_id` bigint(20) NOT NULL,
            `mobile_number` varchar(20) DEFAULT NULL,
            `license_key` text DEFAULT NULL,
            `is_verified` tinyint(1) DEFAULT 0,
            `user_state` varchar(50) DEFAULT 'new',
            `otp_code` varchar(255) DEFAULT NULL,
            `otp_created_at` timestamp NULL DEFAULT NULL,
            `verified_at` timestamp NULL DEFAULT NULL,
            `last_activity` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
            `payment_id` int(11) DEFAULT NULL,
            `subscription_end` timestamp NULL DEFAULT NULL,
            `login_attempts` int(11) DEFAULT 0,
            `is_blocked` tinyint(1) DEFAULT 0,
            `admin_notes` text DEFAULT NULL,
            PRIMARY KEY (`id`),
            UNIQUE KEY `user_id` (`user_id`),
            KEY `idx_username` (`username`),
            KEY `idx_mobile` (`mobile_number`),
            KEY `idx_verified` (`is_verified`),
            KEY `idx_state` (`user_state`),
            KEY `idx_blocked` (`is_blocked`),
            KEY `idx_last_activity` (`last_activity`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    }
    
    /**
     * 📋 SQL جدول فعالیت‌های کاربران
     */
    private function getUserActivitiesTableSQL() {
        return "CREATE TABLE IF NOT EXISTS `user_activities` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `user_id` bigint(20) NOT NULL,
            `username` varchar(255) DEFAULT NULL,
            `first_name` varchar(255) DEFAULT NULL,
            `action` text NOT NULL,
            `ip_address` varchar(45) DEFAULT NULL,
            `user_agent` text DEFAULT NULL,
            `additional_data` json DEFAULT NULL,
            `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_user_id` (`user_id`),
            KEY `idx_created_at` (`created_at`),
            KEY `idx_action` (`action`(100))
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    }
    
    /**
     * 📚 SQL جدول منوهای دانشجویی
     */
    private function getStudentMenusTableSQL() {
        return "CREATE TABLE IF NOT EXISTS `student_menus` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `title` varchar(255) NOT NULL,
            `content` longtext NOT NULL,
            `content_type` enum('text','link','file','video','channel','faq') DEFAULT 'text',
            `icon` varchar(50) DEFAULT '📋',
            `sort_order` int(11) DEFAULT 0,
            `is_active` tinyint(1) DEFAULT 1,
            `access_count` int(11) DEFAULT 0,
            `requires_subscription` tinyint(1) DEFAULT 1,
            `admin_only` tinyint(1) DEFAULT 0,
            `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_active_order` (`is_active`, `sort_order`),
            KEY `idx_requires_sub` (`requires_subscription`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    }
    
    /**
     * 💳 SQL جدول پرداخت‌ها
     */
    private function getPaymentsTableSQL() {
        return "CREATE TABLE IF NOT EXISTS `payments` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `user_id` bigint(20) NOT NULL,
            `amount` decimal(12,2) NOT NULL,
            `payment_method` enum('zarinpal','card_to_card','crypto','admin','free') NOT NULL,
            `transaction_id` varchar(255) DEFAULT NULL,
            `ref_id` varchar(255) DEFAULT NULL,
            `status` enum('pending','completed','failed','cancelled','reviewing','rejected') DEFAULT 'pending',
            `receipt_image` text DEFAULT NULL,
            `admin_notes` text DEFAULT NULL,
            `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            `verified_by` varchar(100) DEFAULT NULL,
            `verification_date` timestamp NULL DEFAULT NULL,
            `callback_data` json DEFAULT NULL,
            PRIMARY KEY (`id`),
            KEY `idx_user_payment` (`user_id`, `status`),
            KEY `idx_transaction` (`transaction_id`),
            KEY `idx_status` (`status`),
            KEY `idx_method` (`payment_method`),
            KEY `idx_created_at` (`created_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    }
    
    /**
     * 🎫 SQL جدول لایسنس‌های SpotPlayer
     */
    private function getSpotPlayerLicensesTableSQL() {
        return "CREATE TABLE IF NOT EXISTS `spotplayer_licenses` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `user_id` bigint(20) NOT NULL,
            `payment_id` int(11) DEFAULT NULL,
            `license_id` varchar(255) NOT NULL,
            `license_key` text NOT NULL,
            `download_url` text DEFAULT NULL,
            `course_ids` json DEFAULT NULL,
            `is_test` tinyint(1) DEFAULT 0,
            `is_active` tinyint(1) DEFAULT 1,
            `device_limit` int(11) DEFAULT 3,
            `current_devices` int(11) DEFAULT 0,
            `expires_at` timestamp NULL DEFAULT NULL,
            `last_used` timestamp NULL DEFAULT NULL,
            `usage_count` int(11) DEFAULT 0,
            `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `license_id` (`license_id`),
            KEY `idx_user_license` (`user_id`, `is_active`),
            KEY `idx_payment` (`payment_id`),
            KEY `idx_expires` (`expires_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    }
    
    /**
     * ⚙️ SQL جدول تنظیمات ادمین
     */
    private function getAdminSettingsTableSQL() {
        return "CREATE TABLE IF NOT EXISTS `admin_settings` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `setting_key` varchar(100) NOT NULL,
            `setting_value` longtext DEFAULT NULL,
            `setting_type` enum('string','integer','boolean','json','text') DEFAULT 'string',
            `description` text DEFAULT NULL,
            `category` varchar(50) DEFAULT 'general',
            `is_public` tinyint(1) DEFAULT 0,
            `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            `updated_by` varchar(100) DEFAULT NULL,
            PRIMARY KEY (`id`),
            UNIQUE KEY `setting_key` (`setting_key`),
            KEY `idx_category` (`category`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    }
    
    /**
     * 📸 SQL جدول رسیدهای پرداخت
     */
    private function getPaymentReceiptsTableSQL() {
        return "CREATE TABLE IF NOT EXISTS `payment_receipts` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `payment_id` int(11) NOT NULL,
            `file_id` varchar(255) NOT NULL,
            `file_url` text DEFAULT NULL,
            `file_type` varchar(50) DEFAULT 'photo',
            `file_size` int(11) DEFAULT NULL,
            `uploaded_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
            `processed` tinyint(1) DEFAULT 0,
            PRIMARY KEY (`id`),
            KEY `idx_payment` (`payment_id`),
            KEY `idx_processed` (`processed`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    }
    
    /**
     * 📄 ایجاد داده‌های پیش‌فرض
     */
    private function insertDefaultData() {
        try {
            // بررسی وجود منوها
            $stmt = $this->connection->prepare("SELECT COUNT(*) as count FROM student_menus");
            $stmt->execute();
            $result = $stmt->fetch();
            
            if ($result['count'] == 0) {
                $this->insertDefaultMenus();
            }
            
            // بررسی وجود تنظیمات
            $stmt = $this->connection->prepare("SELECT COUNT(*) as count FROM admin_settings");
            $stmt->execute();
            $result = $stmt->fetch();
            
            if ($result['count'] == 0) {
                $this->insertDefaultSettings();
            }
            
        } catch (Exception $e) {
            error_log("Error inserting default data: " . $e->getMessage());
        }
    }
    
    /**
     * 📚 ایجاد منوهای پیش‌فرض
     */
    private function insertDefaultMenus() {
        $default_menus = [
            [
                'title' => 'ویدیوهای آموزشی',
                'content' => "🎬 <b>کتابخانه ویدیوهای آموزشی</b>\n\n━━━━━━━━━━━━━━━━━━━━\n\n📚 <b>برای مشاهده ویدیوها:</b>\n\n1️⃣ اپلیکیشن SpotPlayer را دانلود کنید\n2️⃣ لایسنس خود را وارد کنید\n3️⃣ از تمام محتواها لذت ببرید\n\n━━━━━━━━━━━━━━━━━━━━\n\n🔗 <b>لینک دانلود:</b>\nhttps://spotplayer.ir\n\n💎 <b>بیش از 100 ساعت آموزش حرفه‌ای!</b>",
                'content_type' => 'text',
                'icon' => '🎬',
                'sort_order' => 1
            ],
            [
                'title' => 'سیگنال‌های معاملاتی',
                'content' => "📊 <b>کانال سیگنال‌های طلایی</b>\n\n━━━━━━━━━━━━━━━━━━━━\n\n🔥 <b>ویژگی‌های کانال:</b>\n\n💎 سیگنال‌های بلندمدت با دقت بالا\n📈 فارکس، بورس، ارز دیجیتال و طلا\n💰 ارزش ماهانه بیش از 100 دلار\n⚡ پشتیبانی 24/7\n\n━━━━━━━━━━━━━━━━━━━━\n\n🎯 <b>برای عضویت:</b>\n" . SUPPORT_USERNAME . "\n\n🏆 <b>نرخ موفقیت بالای 85%</b>",
                'content_type' => 'text',
                'icon' => '📊',
                'sort_order' => 2
            ],
            [
                'title' => 'گروه VIP دانشجویان',
                'content' => "💎 <b>گروه اختصاصی دانشجویان</b>\n\n━━━━━━━━━━━━━━━━━━━━\n\n👥 <b>امکانات گروه:</b>\n\n🎓 20 ادمین از دانشجویان موفق\n💬 رفع اشکال مستقیم\n📈 تجربه‌های معاملاتی\n⚡ انگیزه و انرژی مثبت\n🎯 چالش‌های هفتگی\n\n━━━━━━━━━━━━━━━━━━━━\n\n📱 <b>برای دعوت به گروه:</b>\nمنتظر پیام دعوت باشید یا با پشتیبانی تماس بگیرید\n\n🔥 <b>جامعه‌ای از معامله‌گران موفق!</b>",
                'content_type' => 'text',
                'icon' => '💎',
                'sort_order' => 3
            ],
            [
                'title' => 'کلوز فرند اینستاگرام',
                'content' => "📸 <b>کلوز فرند اختصاصی کاپیتان</b>\n\n━━━━━━━━━━━━━━━━━━━━\n\n🔴 <b>محتوای اختصاصی:</b>\n\n🎥 لایو ترید روزانه\n📚 آموزش‌های کوتاه و مفید\n📊 تحلیل بازار لحظه‌ای\n💬 پاسخ سوالات مستقیم\n🎯 نکات طلایی معاملات\n\n━━━━━━━━━━━━━━━━━━━━\n\n📸 <b>اینستاگرام:</b> " . INSTAGRAM_PAGE . "\n\n💌 <b>برای عضویت:</b>\nپیام «دوره» را در دایرکت ارسال کنید\n\n🌟 <b>محتوای روزانه و آپدیت مداوم!</b>",
                'content_type' => 'text',
                'icon' => '📸',
                'sort_order' => 4
            ],
            [
                'title' => 'سوالات متداول',
                'content' => "❓ <b>پاسخ سوالات رایج</b>\n\n━━━━━━━━━━━━━━━━━━━━\n\n🔍 <b>سوالات پرتکرار:</b>\n\n❓ <b>چطور لایسنس را فعال کنم؟</b>\n✅ لایسنس را در اپلیکیشن SpotPlayer وارد کنید\n\n❓ <b>ویدیوها کجا هستند؟</b>\n✅ تمام ویدیوها در اپلیکیشن SpotPlayer\n\n❓ <b>چطور به گروه VIP بپیوندم؟</b>\n✅ بعد از فعال‌سازی، لینک ارسال می‌شود\n\n❓ <b>آیا پشتیبانی دارید؟</b>\n✅ پشتیبانی 24/7: " . SUPPORT_USERNAME . "\n\n━━━━━━━━━━━━━━━━━━━━\n\n🤔 <b>سوال دیگری دارید؟</b>\nبا پشتیبانی در تماس باشید!",
                'content_type' => 'faq',
                'icon' => '❓',
                'sort_order' => 5
            ],
            [
                'title' => 'پشتیبانی اختصاصی',
                'content' => "📞 <b>پشتیبانی 24 ساعته</b>\n\n━━━━━━━━━━━━━━━━━━━━\n\n🎯 <b>راه‌های ارتباط:</b>\n\n📱 <b>تلگرام (پاسخ فوری):</b>\n" . SUPPORT_USERNAME . "\n\n📸 <b>اینستاگرام:</b>\n" . INSTAGRAM_PAGE . "\n\n━━━━━━━━━━━━━━━━━━━━\n\n⏰ <b>ساعات پاسخگویی:</b>\n🕐 8 صبح تا 12 شب\n\n🚀 <b>زمان پاسخ معمولی:</b>\nکمتر از 30 دقیقه\n\n━━━━━━━━━━━━━━━━━━━━\n\n💡 <b>برای پاسخ سریع‌تر:</b>\n• سوال کامل و واضح بپرسید\n• اسکرین‌شات یا عکس بفرستید\n• صبور باشید، پاسخ حتمی است\n\n🔥 <b>تیم پشتیبانی در خدمت شماست!</b>",
                'content_type' => 'text',
                'icon' => '📞',
                'sort_order' => 6
            ]
        ];
        
        $stmt = $this->connection->prepare("
            INSERT INTO student_menus (title, content, content_type, icon, sort_order) 
            VALUES (?, ?, ?, ?, ?)
        ");
        
        foreach ($default_menus as $menu) {
            try {
                $stmt->execute([
                    $menu['title'],
                    $menu['content'],
                    $menu['content_type'],
                    $menu['icon'],
                    $menu['sort_order']
                ]);
                error_log("Inserted default menu: " . $menu['title']);
            } catch (Exception $e) {
                error_log("Error inserting menu '{$menu['title']}': " . $e->getMessage());
            }
        }
    }
    
    /**
     * ⚙️ ایجاد تنظیمات پیش‌فرض
     */
    private function insertDefaultSettings() {
        $default_settings = [
            ['maintenance_mode', 'false', 'boolean', 'حالت تعمیرات سیستم', 'system'],
            ['welcome_message_active', 'true', 'boolean', 'نمایش پیام خوشامدگویی', 'messages'],
            ['auto_license_creation', 'true', 'boolean', 'ایجاد خودکار لایسنس', 'spotplayer'],
            ['payment_card_active', 'true', 'boolean', 'فعال بودن پرداخت کارت به کارت', 'payment'],
            ['payment_zarinpal_active', 'false', 'boolean', 'فعال بودن پرداخت زرین‌پال', 'payment'],
            ['course_discount_percent', '0', 'integer', 'درصد تخفیف دوره', 'course'],
            ['max_login_attempts', '5', 'integer', 'حداکثر تلاش ورود', 'security'],
            ['otp_expiry_minutes', '5', 'integer', 'مدت اعتبار کد تایید (دقیقه)', 'security'],
            ['admin_notifications', 'true', 'boolean', 'اعلان‌های ادمین', 'admin'],
            ['auto_verify_payments', 'false', 'boolean', 'تایید خودکار پرداخت‌ها', 'payment']
        ];
        
        $stmt = $this->connection->prepare("
            INSERT INTO admin_settings (setting_key, setting_value, setting_type, description, category) 
            VALUES (?, ?, ?, ?, ?)
        ");
        
        foreach ($default_settings as $setting) {
            try {
                $stmt->execute($setting);
                error_log("Inserted default setting: " . $setting[0]);
            } catch (Exception $e) {
                error_log("Error inserting setting '{$setting[0]}': " . $e->getMessage());
            }
        }
    }
    
    /**
     * 👤 ثبت/به‌روزرسانی کاربر
     */
    public function registerUser($user_id, $username, $first_name, $chat_id) {
        try {
            $stmt = $this->connection->prepare("
                INSERT INTO users (user_id, username, first_name, chat_id, created_at, last_activity) 
                VALUES (?, ?, ?, ?, NOW(), NOW()) 
                ON DUPLICATE KEY UPDATE 
                username = VALUES(username), 
                first_name = VALUES(first_name),
                chat_id = VALUES(chat_id),
                last_activity = NOW()
            ");
            
            return $stmt->execute([$user_id, $username, $first_name, $chat_id]);
            
        } catch (Exception $e) {
            error_log("Register user error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🎯 تنظیم وضعیت کاربر
     */
    public function setUserState($user_id, $state) {
        try {
            $stmt = $this->connection->prepare("
                UPDATE users SET user_state = ?, last_activity = NOW() WHERE user_id = ?
            ");
            
            return $stmt->execute([$state, $user_id]);
            
        } catch (Exception $e) {
            error_log("Set user state error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🔍 دریافت وضعیت کاربر
     */
    public function getUserState($user_id) {
        try {
            $stmt = $this->connection->prepare("SELECT user_state FROM users WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $result = $stmt->fetch();
            
            return $result ? $result['user_state'] : 'new';
            
        } catch (Exception $e) {
            error_log("Get user state error: " . $e->getMessage());
            return 'new';
        }
    }
    
    /**
     * ℹ️ دریافت اطلاعات کاربر
     */
    public function getUserInfo($user_id) {
        try {
            $stmt = $this->connection->prepare("
                SELECT * FROM users WHERE user_id = ?
            ");
            $stmt->execute([$user_id]);
            return $stmt->fetch();
            
        } catch (Exception $e) {
            error_log("Get user info error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🎫 تنظیم لایسنس کاربر
     */
    public function setUserLicense($user_id, $license) {
        try {
            $stmt = $this->connection->prepare("
                UPDATE users SET 
                license_key = ?, 
                is_verified = 1, 
                verified_at = NOW(),
                user_state = 'authenticated'
                WHERE user_id = ?
            ");
            
            return $stmt->execute([$license, $user_id]);
            
        } catch (Exception $e) {
            error_log("Set user license error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🔐 تنظیم کد OTP
     */
    public function setUserOTP($user_id, $otp, $mobile) {
        try {
            $hashed_otp = password_hash($otp, PASSWORD_DEFAULT);
            
            $stmt = $this->connection->prepare("
                UPDATE users SET 
                otp_code = ?, 
                mobile_number = ?, 
                otp_created_at = NOW() 
                WHERE user_id = ?
            ");
            
            return $stmt->execute([$hashed_otp, $mobile, $user_id]);
            
        } catch (Exception $e) {
            error_log("Set user OTP error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * ✅ تایید کد OTP
     */
    public function verifyOTP($user_id, $otp) {
        try {
            $stmt = $this->connection->prepare("
                SELECT otp_code, otp_created_at FROM users 
                WHERE user_id = ? 
                AND otp_created_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE)
            ");
            
            $stmt->execute([$user_id]);
            $result = $stmt->fetch();
            
            if ($result && password_verify($otp, $result['otp_code'])) {
                // Mark as verified
                $this->connection->prepare("
                    UPDATE users SET 
                    is_verified = 1, 
                    verified_at = NOW(), 
                    otp_code = NULL,
                    user_state = 'authenticated'
                    WHERE user_id = ?
                ")->execute([$user_id]);
                
                return true;
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("Verify OTP error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 📋 ثبت فعالیت کاربر
     */
    public function logUserActivity($user_id, $username, $first_name, $action, $additional_data = null) {
        try {
            $ip_address = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown';
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
            
            $stmt = $this->connection->prepare("
                INSERT INTO user_activities 
                (user_id, username, first_name, action, ip_address, user_agent, additional_data, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            
            return $stmt->execute([
                $user_id, 
                $username, 
                $first_name, 
                $action, 
                $ip_address, 
                $user_agent,
                $additional_data ? json_encode($additional_data) : null
            ]);
            
        } catch (Exception $e) {
            error_log("Log activity error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 📚 دریافت منوهای دانشجویی
     */
    public function getStudentMenus() {
        try {
            $stmt = $this->connection->prepare("
                SELECT * FROM student_menus 
                WHERE is_active = 1 
                ORDER BY sort_order ASC, created_at ASC
            ");
            
            $stmt->execute();
            return $stmt->fetchAll();
            
        } catch (Exception $e) {
            error_log("Get student menus error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * 📖 دریافت محتوای منو
     */
    public function getMenuContent($menu_id) {
        try {
            $stmt = $this->connection->prepare("
                SELECT * FROM student_menus WHERE id = ? AND is_active = 1
            ");
            
            $stmt->execute([$menu_id]);
            $result = $stmt->fetch();
            
            if ($result) {
                // Increment access count
                $this->connection->prepare("
                    UPDATE student_menus SET access_count = access_count + 1 WHERE id = ?
                ")->execute([$menu_id]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log("Get menu content error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🛒 بررسی خرید قبلی کاربر
     */
    public function hasUserPurchased($user_id) {
        try {
            $stmt = $this->connection->prepare("
                SELECT COUNT(*) as count FROM payments 
                WHERE user_id = ? AND status = 'completed'
            ");
            
            $stmt->execute([$user_id]);
            $result = $stmt->fetch();
            
            return $result['count'] > 0;
            
        } catch (Exception $e) {
            error_log("Check purchase error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 💳 ایجاد رکورد پرداخت
     */
    public function createPayment($user_id, $amount, $payment_method, $transaction_id = null) {
        try {
            $stmt = $this->connection->prepare("
                INSERT INTO payments (user_id, amount, payment_method, transaction_id, status, created_at) 
                VALUES (?, ?, ?, ?, 'pending', NOW())
            ");
            
            $stmt->execute([$user_id, $amount, $payment_method, $transaction_id]);
            return $this->connection->lastInsertId();
            
        } catch (Exception $e) {
            error_log("Create payment error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🆔 تنظیم شناسه پرداخت کاربر
     */
    public function setUserPaymentId($user_id, $payment_id) {
        try {
            $stmt = $this->connection->prepare("
                UPDATE users SET payment_id = ? WHERE user_id = ?
            ");
            
            return $stmt->execute([$payment_id, $user_id]);
            
        } catch (Exception $e) {
            error_log("Set payment ID error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 📸 به‌روزرسانی رسید پرداخت
     */
    public function updatePaymentReceipt($payment_id, $receipt_url) {
        try {
            $stmt = $this->connection->prepare("
                UPDATE payments SET 
                receipt_image = ?, 
                status = 'reviewing',
                updated_at = NOW()
                WHERE id = ?
            ");
            
            return $stmt->execute([$receipt_url, $payment_id]);
            
        } catch (Exception $e) {
            error_log("Update payment receipt error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🔄 به‌روزرسانی وضعیت پرداخت
     */
    public function updatePaymentStatus($payment_id, $status, $ref_id = null, $verified_by = null) {
        try {
            $stmt = $this->connection->prepare("
                UPDATE payments SET 
                status = ?, 
                ref_id = ?, 
                updated_at = NOW(),
                verified_by = ?,
                verification_date = CASE WHEN ? IN ('completed', 'rejected') THEN NOW() ELSE verification_date END
                WHERE id = ?
            ");
            
            return $stmt->execute([$status, $ref_id, $verified_by, $status, $payment_id]);
            
        } catch (Exception $e) {
            error_log("Update payment status error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 💳 دریافت پرداخت‌های در انتظار
     */
    public function getPendingPayments($limit = 20) {
        try {
            $stmt = $this->connection->prepare("
                SELECT p.*, u.first_name, u.username, u.mobile_number, u.chat_id
                FROM payments p
                JOIN users u ON p.user_id = u.user_id
                WHERE p.status IN ('pending', 'reviewing')
                ORDER BY p.created_at ASC
                LIMIT ?
            ");
            
            $stmt->execute([$limit]);
            return $stmt->fetchAll();
            
        } catch (Exception $e) {
            error_log("Get pending payments error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * ℹ️ دریافت اطلاعات پرداخت
     */
    public function getPaymentInfo($payment_id) {
        try {
            $stmt = $this->connection->prepare("
                SELECT p.*, u.first_name, u.username, u.mobile_number, u.chat_id
                FROM payments p
                JOIN users u ON p.user_id = u.user_id
                WHERE p.id = ?
            ");
            
            $stmt->execute([$payment_id]);
            return $stmt->fetch();
            
        } catch (Exception $e) {
            error_log("Get payment info error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 📊 آمار ادمین
     */
    public function getAdminStats() {
        $stats = [];
        
        try {
            // کل کاربران
            $stmt = $this->connection->query("SELECT COUNT(*) as count FROM users");
            $stats['total_users'] = $stmt->fetch()['count'];
            
            // کاربران تایید شده
            $stmt = $this->connection->query("SELECT COUNT(*) as count FROM users WHERE is_verified = 1");
            $stats['verified_users'] = $stmt->fetch()['count'];
            
            // آمار پرداخت‌ها
            $stmt = $this->connection->query("
                SELECT COUNT(*) as count, IFNULL(SUM(amount), 0) as total 
                FROM payments WHERE status = 'completed'
            ");
            $payment_stats = $stmt->fetch();
            $stats['total_payments'] = $payment_stats['count'];
            $stats['total_revenue'] = $payment_stats['total'];
            
            // ثبت‌نام امروز
            $stmt = $this->connection->query("
                SELECT COUNT(*) as count FROM users WHERE DATE(created_at) = CURDATE()
            ");
            $stats['today_registrations'] = $stmt->fetch()['count'];
            
            // پرداخت‌های در انتظار
            $stmt = $this->connection->query("
                SELECT COUNT(*) as count FROM payments WHERE status IN ('pending', 'reviewing')
            ");
            $stats['pending_payments'] = $stmt->fetch()['count'];
            
            // رشد هفتگی
            $stmt = $this->connection->query("
                SELECT COUNT(*) as count FROM users 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
            ");
            $stats['weekly_growth'] = $stmt->fetch()['count'];
            
            // محبوب‌ترین منوها
            $stmt = $this->connection->query("
                SELECT title, access_count FROM student_menus 
                WHERE is_active = 1 
                ORDER BY access_count DESC 
                LIMIT 3
            ");
            $stats['popular_menus'] = $stmt->fetchAll();
            
        } catch (Exception $e) {
            error_log("Get admin stats error: " . $e->getMessage());
        }
        
        return $stats;
    }
    
    /**
     * 📋 فعالیت‌های اخیر
     */
    public function getRecentActivities($limit = 15) {
        try {
            $stmt = $this->connection->prepare("
                SELECT * FROM user_activities 
                ORDER BY created_at DESC 
                LIMIT ?
            ");
            
            $stmt->execute([$limit]);
            return $stmt->fetchAll();
            
        } catch (Exception $e) {
            error_log("Get recent activities error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * 📖 ثبت دسترسی به منو
     */
    public function logMenuAccess($user_id, $menu_id) {
        try {
            // Update menu access count
            $stmt = $this->connection->prepare("
                UPDATE student_menus SET access_count = access_count + 1 WHERE id = ?
            ");
            $stmt->execute([$menu_id]);
            
            // Log activity
            $stmt = $this->connection->prepare("
                INSERT INTO user_activities (user_id, username, first_name, action, created_at)
                SELECT ?, username, first_name, CONCAT('دسترسی به منو: ', ?), NOW()
                FROM users WHERE user_id = ?
            ");
            
            return $stmt->execute([$user_id, $menu_id, $user_id]);
            
        } catch (Exception $e) {
            error_log("Log menu access error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🎫 ایجاد رکورد لایسنس SpotPlayer
     */
    public function createSpotPlayerLicense($user_id, $payment_id, $license_data) {
        try {
            $stmt = $this->connection->prepare("
                INSERT INTO spotplayer_licenses 
                (user_id, payment_id, license_id, license_key, download_url, course_ids, is_test, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            
            return $stmt->execute([
                $user_id,
                $payment_id,
                $license_data['license_id'],
                $license_data['license_key'],
                $license_data['download_url'] ?? null,
                json_encode($license_data['course_ids'] ?? []),
                $license_data['is_test'] ? 1 : 0
            ]);
            
        } catch (Exception $e) {
            error_log("Create SpotPlayer license error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🔍 جستجوی کاربران
     */
    public function searchUsers($term, $limit = 50) {
        try {
            $search = "%$term%";
            $stmt = $this->connection->prepare("
                SELECT * FROM users 
                WHERE first_name LIKE ? 
                   OR username LIKE ? 
                   OR mobile_number LIKE ?
                   OR user_id LIKE ?
                ORDER BY last_activity DESC
                LIMIT ?
            ");
            
            $stmt->execute([$search, $search, $search, $search, $limit]);
            return $stmt->fetchAll();
            
        } catch (Exception $e) {
            error_log("Search users error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * 🔄 تغییر وضعیت مسدودیت کاربر
     */
    public function toggleUserBlock($user_id, $blocked) {
        try {
            $stmt = $this->connection->prepare("
                UPDATE users SET 
                is_blocked = ?, 
                last_activity = NOW() 
                WHERE user_id = ?
            ");
            
            return $stmt->execute([$blocked ? 1 : 0, $user_id]);
            
        } catch (Exception $e) {
            error_log("Toggle user block error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 🏥 وضعیت سلامت سیستم
     */
    public function getSystemHealth() {
        $health = [
            'database' => 'ok',
            'tables' => [],
            'performance' => 'good',
            'last_activity' => null
        ];
        
        try {
            $tables = ['users', 'payments', 'student_menus', 'user_activities'];
            foreach ($tables as $table) {
                $stmt = $this->connection->query("SELECT COUNT(*) as count FROM $table");
                $health['tables'][$table] = $stmt->fetch()['count'];
            }
            
            // آخرین فعالیت
            $stmt = $this->connection->query("SELECT MAX(created_at) as last_activity FROM user_activities");
            $health['last_activity'] = $stmt->fetch()['last_activity'];
            
            // Performance check
            $start_time = microtime(true);
            $this->connection->query("SELECT 1");
            $query_time = microtime(true) - $start_time;
            
            if ($query_time > 1) {
                $health['performance'] = 'slow';
            } elseif ($query_time > 0.5) {
                $health['performance'] = 'moderate';
            }
            
        } catch (Exception $e) {
            $health['database'] = 'error';
            $health['error'] = $e->getMessage();
        }
        
        return $health;
    }
    
    /**
     * 🔒 شروع تراکنش
     */
    public function beginTransaction() {
        if (!$this->transaction_active) {
            $this->connection->beginTransaction();
            $this->transaction_active = true;
        }
    }
    
    /**
     * ✅ تایید تراکنش
     */
    public function commit() {
        if ($this->transaction_active) {
            $this->connection->commit();
            $this->transaction_active = false;
        }
    }
    
    /**
     * ❌ لغو تراکنش
     */
    public function rollback() {
        if ($this->transaction_active) {
            $this->connection->rollback();
            $this->transaction_active = false;
        }
    }
    
    /**
     * 🔄 بستن اتصال
     */
    public function __destruct() {
        if ($this->transaction_active) {
            $this->rollback();
        }
    }
}
?>