<?php
/**
 * Callback Query Handler System
 * Updated: 2025-10-15 19:15:26 UTC
 * Created by: saportinsta65
 */

class CallbackHandler {
    private $telegram;
    private $db;
    
    public function __construct($telegram, $db) {
        $this->telegram = $telegram;
        $this->db = $db;
    }
    
    /**
     * 🎯 مدیریت کال‌بک‌های ورودی
     */
    public function handleCallback($callback_id, $chat_id, $user_id, $data) {
        // Answer callback first
        $this->telegram->answerCallbackQuery($callback_id, "در حال پردازش...");
        
        // Parse callback data
        $parts = explode('_', $data);
        $action = $parts[0] ?? '';
        
        switch ($action) {
            case 'payment':
                $this->handlePaymentCallback($chat_id, $user_id, $parts);
                break;
                
            case 'admin':
                $this->handleAdminCallback($chat_id, $user_id, $parts);
                break;
                
            case 'menu':
                $this->handleMenuCallback($chat_id, $user_id, $parts);
                break;
                
            case 'course':
                $this->handleCourseCallback($chat_id, $user_id, $parts);
                break;
                
            default:
                $this->handleUnknownCallback($chat_id, $user_id, $data);
                break;
        }
    }
    
    /**
     * 💳 کال‌بک‌های پرداخت
     */
    private function handlePaymentCallback($chat_id, $user_id, $parts) {
        $sub_action = $parts[1] ?? '';
        $payment_id = $parts[2] ?? '';
        
        switch ($sub_action) {
            case 'retry':
                $this->retryPayment($chat_id, $user_id, $payment_id);
                break;
                
            case 'cancel':
                $this->cancelPayment($chat_id, $user_id, $payment_id);
                break;
                
            case 'status':
                $this->checkPaymentStatus($chat_id, $user_id, $payment_id);
                break;
                
            default:
                $this->telegram->sendMessage($chat_id, "❌ عملیات نامعتبر");
                break;
        }
    }
    
    /**
     * 👨‍💼 کال‌بک‌های ادمین
     */
    private function handleAdminCallback($chat_id, $user_id, $parts) {
        // Check admin access
        if ($user_id != ADMIN_CHAT_ID) {
            $this->telegram->sendMessage($chat_id, "❌ دسترسی غیرمجاز");
            return;
        }
        
        $sub_action = $parts[1] ?? '';
        $target_id = $parts[2] ?? '';
        
        switch ($sub_action) {
            case 'approve':
                $this->adminApprovePayment($chat_id, $user_id, $target_id);
                break;
                
            case 'reject':
                $this->adminRejectPayment($chat_id, $user_id, $target_id);
                break;
                
            case 'user':
                $this->adminManageUser($chat_id, $user_id, $target_id);
                break;
                
            case 'broadcast':
                $this->adminBroadcastConfirm($chat_id, $user_id);
                break;
                
            default:
                $this->telegram->sendMessage($chat_id, "❌ عملیات ادمین نامعتبر");
                break;
        }
    }
    
    /**
     * 📚 کال‌بک‌های منو
     */
    private function handleMenuCallback($chat_id, $user_id, $parts) {
        $menu_id = $parts[1] ?? '';
        
        if (!$menu_id) {
            $this->telegram->sendMessage($chat_id, "❌ منوی نامعتبر");
            return;
        }
        
        // Check user verification
        $user_info = $this->db->getUserInfo($user_id);
        if (!$user_info || !$user_info['is_verified']) {
            $this->telegram->sendMessage($chat_id, "🔐 ابتدا باید وارد شوید.\n\n🎓 از «ورود دانشجویان» استفاده کنید.");
            return;
        }
        
        // Get menu content
        $menu = $this->db->getMenuContent($menu_id);
        if (!$menu) {
            $this->telegram->sendMessage($chat_id, "❌ منو یافت نشد");
            return;
        }
        
        // Send content
        $this->telegram->sendMessage($chat_id, $menu['content']);
        
        // Log access
        $this->db->logMenuAccess($user_id, $menu_id);
    }
    
    /**
     * 📚 کال‌بک‌های دوره
     */
    private function handleCourseCallback($chat_id, $user_id, $parts) {
        $sub_action = $parts[1] ?? '';
        
        switch ($sub_action) {
            case 'info':
                $this->showCourseInfo($chat_id, $user_id);
                break;
                
            case 'buy':
                $this->startCoursePurchase($chat_id, $user_id);
                break;
                
            case 'preview':
                $this->showCoursePreview($chat_id, $user_id);
                break;
                
            default:
                $this->telegram->sendMessage($chat_id, "❌ عملیات دوره نامعتبر");
                break;
        }
    }
    
    /**
     * 🔄 تلاش مجدد پرداخت
     */
    private function retryPayment($chat_id, $user_id, $payment_id) {
        try {
            $payment = $this->db->getPaymentInfo($payment_id);
            
            if (!$payment || $payment['user_id'] != $user_id) {
                $this->telegram->sendMessage($chat_id, "❌ پرداخت یافت نشد");
                return;
            }
            
            if ($payment['status'] === 'completed') {
                $this->telegram->sendMessage($chat_id, "✅ این پرداخت قبلاً تکمیل شده است");
                return;
            }
            
            // Reset payment state
            $this->db->setUserState($user_id, 'awaiting_receipt');
            $this->db->setUserPaymentId($user_id, $payment_id);
            
            $message = "🔄 <b>تلاش مجدد پرداخت</b>

━━━━━━━━━━━━━━━━━━━━

🆔 کد: <code>{$payment_id}</code>
💰 مبلغ: " . number_format($payment['amount']) . " تومان

📸 <b>لطفاً رسید جدید یا واضح‌تری ارسال کنید</b>";
            
            $this->telegram->sendMessage($chat_id, $message);
            
        } catch (Exception $e) {
            error_log("Retry payment error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در تلاش مجدد");
        }
    }
    
    /**
     * ❌ لغو پرداخت
     */
    private function cancelPayment($chat_id, $user_id, $payment_id) {
        try {
            $payment = $this->db->getPaymentInfo($payment_id);
            
            if (!$payment || $payment['user_id'] != $user_id) {
                $this->telegram->sendMessage($chat_id, "❌ پرداخت یافت نشد");
                return;
            }
            
            if ($payment['status'] === 'completed') {
                $this->telegram->sendMessage($chat_id, "❌ پرداخت تکمیل شده قابل لغو نیست");
                return;
            }
            
            // Cancel payment
            $this->db->updatePaymentStatus($payment_id, 'cancelled');
            $this->db->setUserState($user_id, 'main_menu');
            
            $message = "❌ <b>پرداخت لغو شد</b>

🆔 کد: <code>{$payment_id}</code>

🔄 <b>می‌توانید دوباره تلاش کنید</b>";
            
            $keyboard = [
                'keyboard' => [
                    [
                        ['text' => '📚 شروع یادگیری'],
                        ['text' => '🏠 منوی اصلی']
                    ]
                ],
                'resize_keyboard' => true
            ];
            
            $this->telegram->sendMessage($chat_id, $message, $keyboard);
            
        } catch (Exception $e) {
            error_log("Cancel payment error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در لغو پرداخت");
        }
    }
    
    /**
     * 🔍 بررسی وضعیت پرداخت
     */
    private function checkPaymentStatus($chat_id, $user_id, $payment_id) {
        try {
            $payment = $this->db->getPaymentInfo($payment_id);
            
            if (!$payment || $payment['user_id'] != $user_id) {
                $this->telegram->sendMessage($chat_id, "❌ پرداخت یافت نشد");
                return;
            }
            
            $status_text = $this->getPaymentStatusText($payment['status']);
            
            $message = "🔍 <b>وضعیت پرداخت</b>

━━━━━━━━━━━━━━━━━━━━

🆔 <b>کد:</b> <code>{$payment_id}</code>
💰 <b>مبلغ:</b> " . number_format($payment['amount']) . " تومان
🔄 <b>وضعیت:</b> {$status_text}
📅 <b>تاریخ:</b> " . date('Y/m/d H:i', strtotime($payment['created_at'])) . "

━━━━━━━━━━━━━━━━━━━━";
            
            if ($payment['status'] === 'pending' || $payment['status'] === 'reviewing') {
                $message .= "

⏰ <b>در حال بررسی...</b>
📞 پشتیبانی: " . SUPPORT_USERNAME;
            } elseif ($payment['status'] === 'completed') {
                $message .= "

✅ <b>پرداخت موفق!</b>
🎫 لایسنس ارسال شده";
            } elseif ($payment['status'] === 'rejected') {
                $message .= "

❌ <b>پرداخت تایید نشد</b>
💡 می‌توانید دوباره تلاش کنید";
            }
            
            $this->telegram->sendMessage($chat_id, $message);
            
        } catch (Exception $e) {
            error_log("Check payment status error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در بررسی وضعیت");
        }
    }
    
    /**
     * 📋 متن وضعیت پرداخت
     */
    private function getPaymentStatusText($status) {
        $statuses = [
            'pending' => '⏳ در انتظار',
            'reviewing' => '🔍 در حال بررسی',
            'completed' => '✅ تکمیل شده',
            'rejected' => '❌ تایید نشد',
            'cancelled' => '🚫 لغو شده',
            'failed' => '💥 ناموفق'
        ];
        
        return $statuses[$status] ?? '❓ نامشخص';
    }
    
    /**
     * ✅ تایید ادمین (کال‌بک)
     */
    private function adminApprovePayment($chat_id, $user_id, $payment_id) {
        $adminCommands = new AdminCommands($this->telegram, $this->db);
        $adminCommands->approvePayment($chat_id, $user_id, $payment_id);
    }
    
    /**
     * ❌ رد ادمین (کال‌بک)
     */
    private function adminRejectPayment($chat_id, $user_id, $payment_id) {
        $adminCommands = new AdminCommands($this->telegram, $this->db);
        $adminCommands->rejectPayment($chat_id, $user_id, $payment_id);
    }
    
    /**
     * 👤 مدیریت کاربر توسط ادمین
     */
    private function adminManageUser($chat_id, $user_id, $target_user_id) {
        try {
            $user = $this->db->getUserInfo($target_user_id);
            
            if (!$user) {
                $this->telegram->sendMessage($chat_id, "❌ کاربر یافت نشد");
                return;
            }
            
            $message = "👤 <b>مدیریت کاربر</b>

━━━━━━━━━━━━━━━━━━━━

👤 <b>نام:</b> {$user['first_name']}
📱 <b>یوزرنیم:</b> @{$user['username']}
🔢 <b>آیدی:</b> <code>{$user['user_id']}</code>
📞 <b>موبایل:</b> {$user['mobile_number']}
✅ <b>تایید:</b> " . ($user['is_verified'] ? 'بله' : 'خیر') . "
🚫 <b>مسدود:</b> " . ($user['is_blocked'] ? 'بله' : 'خیر') . "
📅 <b>عضویت:</b> " . date('Y/m/d', strtotime($user['created_at'])) . "

━━━━━━━━━━━━━━━━━━━━

🎯 <b>اقدامات:</b>";
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        [
                            'text' => $user['is_blocked'] ? '✅ رفع مسدودیت' : '🚫 مسدود کردن',
                            'callback_data' => 'admin_toggle_block_' . $user['user_id']
                        ]
                    ],
                    [
                        [
                            'text' => '🎫 ایجاد لایسنس رایگان',
                            'callback_data' => 'admin_free_license_' . $user['user_id']
                        ]
                    ],
                    [
                        [
                            'text' => '💬 ارسال پیام',
                            'callback_data' => 'admin_message_' . $user['user_id']
                        ]
                    ]
                ]
            ];
            
            $this->telegram->sendMessage($chat_id, $message, $keyboard);
            
        } catch (Exception $e) {
            error_log("Admin manage user error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در مدیریت کاربر");
        }
    }
    
    /**
     * 📚 نمایش اطلاعات دوره
     */
    private function showCourseInfo($chat_id, $user_id) {
        $message = "📚 <b>دوره جامع کاپیتان ترید</b>

━━━━━━━━━━━━━━━━━━━━

🎯 <b>محتوای دوره:</b>

📈 <b>تحلیل تکنیکال:</b>
• مبانی و پیشرفته
• الگوهای قیمتی
• اندیکاتورهای حرفه‌ای

🧠 <b>روانشناسی معاملات:</b>
• کنترل احساسات
• مدیریت ترس و طمع
• انضباط معاملاتی

💰 <b>مدیریت ریسک:</b>
• تعیین حجم موقعیت
• استاپ لاس و تیک پرافیت
• نسبت ریسک به سود

🎯 <b>سیستم PLS:</b>
• استراتژی اختصاصی
• نقاط ورود و خروج
• بک‌تست و نتایج

━━━━━━━━━━━━━━━━━━━━

💎 <b>قیمت:</b> " . number_format(COURSE_PRICE) . " تومان

🎁 <b>بونوس‌ها:</b>
🔔 کانال سیگنال VIP
👥 گروه دانشجویان
📞 پشتیبانی اختصاصی";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => '🛒 خرید دوره',
                        'callback_data' => 'course_buy'
                    ],
                    [
                        'text' => '🎬 پیش‌نمایش',
                        'callback_data' => 'course_preview'
                    ]
                ]
            ]
        ];
        
        $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 🛒 شروع خرید دوره
     */
    private function startCoursePurchase($chat_id, $user_id) {
        // Check if already purchased
        if ($this->db->hasUserPurchased($user_id)) {
            $this->telegram->sendMessage($chat_id, "✅ شما قبلاً این دوره را خریداری کرده‌اید!");
            return;
        }
        
        $message = "🛒 <b>خرید دوره کاپیتان ترید</b>

━━━━━━━━━━━━━━━━━━━━

💰 <b>قیمت:</b> " . number_format(COURSE_PRICE) . " تومان

💳 <b>روش‌های پرداخت:</b>";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => '💳 پرداخت آنلاین',
                        'callback_data' => 'payment_online'
                    ]
                ],
                [
                    [
                        'text' => '💰 کارت به کارت',
                        'callback_data' => 'payment_card'
                    ]
                ]
            ]
        ];
        
        $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 🎬 پیش‌نمایش دوره
     */
    private function showCoursePreview($chat_id, $user_id) {
        $message = "🎬 <b>پیش‌نمایش دوره</b>

━━━━━━━━━━━━━━━━━━━━

📺 <b>ویدیوهای نمونه:</b>

🎯 مقدمه‌ای بر تحلیل تکنیکال
🔥 اولین معامله موفق
💡 اشتباهات رایج تازه‌کارها

━━━━━━━━━━━━━━━━━━━━

🔗 <b>لینک‌های پیش‌نمایش:</b>

📱 یوتیوب: youtube.com/c/captaintrade
📸 اینستاگرام: " . INSTAGRAM_PAGE . "

━━━━━━━━━━━━━━━━━━━━

🎯 <b>آماده شروع یادگیری؟</b>";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => '🛒 خرید دوره کامل',
                        'callback_data' => 'course_buy'
                    ]
                ]
            ]
        ];
        
        $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * ❓ کال‌بک نامشخص
     */
    private function handleUnknownCallback($chat_id, $user_id, $data) {
        $this->telegram->sendMessage($chat_id, "❌ عملیات نامعتبر: {$data}");
        error_log("Unknown callback data: {$data} from user: {$user_id}");
    }
}
?>