<?php
/**
 * Enhanced Security Management System - FIXED VERSION
 * Captain Trade Bot - Professional Security Layer
 * Updated: 2025-10-15 18:35:00 UTC
 * Created by: saportinsta65
 */

class Security {
    private $max_requests_per_minute = 30;
    private $max_requests_per_hour = 200;
    private $max_login_attempts = 5;
    private $block_duration = 3600; // 1 hour
    private $redis_available = false;
    
    public function __construct() {
        // Check if Redis is available for caching
        $this->redis_available = extension_loaded('redis') && class_exists('Redis');
    }
    
    /**
     * Rate limiting check
     */
    public function checkRateLimit($user_id = null) {
        $ip = $this->getClientIP();
        $identifier = $user_id ?: $ip;
        
        // Check minute-based rate limit
        if ($this->isRateLimited($identifier, 'minute', $this->max_requests_per_minute)) {
            $this->logSecurityEvent('rate_limit_exceeded', $identifier, 'minute');
            http_response_code(429);
            exit('Rate limit exceeded. Please slow down.');
        }
        
        // Check hour-based rate limit
        if ($this->isRateLimited($identifier, 'hour', $this->max_requests_per_hour)) {
            $this->logSecurityEvent('rate_limit_exceeded', $identifier, 'hour');
            http_response_code(429);
            exit('Hourly rate limit exceeded. Please try again later.');
        }
        
        // Increment counters
        $this->incrementRateLimit($identifier);
    }
    
    /**
     * Validate user for security threats
     */
    public function validateUser($user_id, $username = null) {
        global $db;
        
        // Check if user is blocked
        if ($this->isUserBlocked($user_id)) {
            $this->logSecurityEvent('blocked_user_attempt', $user_id);
            return false;
        }
        
        // Check for suspicious patterns
        if ($this->isSuspiciousActivity($user_id)) {
            $this->logSecurityEvent('suspicious_activity', $user_id);
            // Don't block immediately, but monitor
        }
        
        // Validate username format
        if ($username && !$this->isValidUsername($username)) {
            $this->logSecurityEvent('invalid_username', $user_id, $username);
            return false;
        }
        
        return true;
    }
    
    /**
     * Get client IP address
     */
    private function getClientIP() {
        $ip_headers = [
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_X_FORWARDED_FOR',      // Proxy
            'HTTP_X_FORWARDED',          // Proxy
            'HTTP_X_CLUSTER_CLIENT_IP',  // Cluster
            'HTTP_FORWARDED_FOR',        // Proxy
            'HTTP_FORWARDED',            // Proxy
            'REMOTE_ADDR'                // Direct connection
        ];
        
        foreach ($ip_headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                
                // Handle comma-separated IPs
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                
                // Validate IP format
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }
    
    /**
     * Check if identifier is rate limited
     */
    private function isRateLimited($identifier, $period, $limit) {
        $key = "rate_limit_{$period}_{$identifier}";
        
        if ($this->redis_available) {
            return $this->checkRateLimitRedis($key, $period, $limit);
        } else {
            return $this->checkRateLimitFile($key, $period, $limit);
        }
    }
    
    /**
     * Redis-based rate limiting (if available)
     */
    private function checkRateLimitRedis($key, $period, $limit) {
        try {
            $redis = new Redis();
            $redis->connect('127.0.0.1', 6379);
            
            $current = $redis->get($key);
            
            if ($current === false) {
                return false; // No limit set yet
            }
            
            return (int)$current >= $limit;
            
        } catch (Exception $e) {
            error_log("Redis rate limit error: " . $e->getMessage());
            return false; // Fail open
        }
    }
    
    /**
     * File-based rate limiting (fallback)
     */
    private function checkRateLimitFile($key, $period, $limit) {
        $file_path = sys_get_temp_dir() . "/rate_limit_$key";
        
        if (!file_exists($file_path)) {
            return false;
        }
        
        $data = json_decode(file_get_contents($file_path), true);
        
        if (!$data || !isset($data['count'], $data['timestamp'])) {
            return false;
        }
        
        $expiry = $period === 'minute' ? 60 : 3600;
        
        // Check if period has expired
        if (time() - $data['timestamp'] > $expiry) {
            unlink($file_path);
            return false;
        }
        
        return $data['count'] >= $limit;
    }
    
    /**
     * Increment rate limit counter
     */
    private function incrementRateLimit($identifier) {
        $periods = ['minute' => 60, 'hour' => 3600];
        
        foreach ($periods as $period => $expiry) {
            $key = "rate_limit_{$period}_{$identifier}";
            
            if ($this->redis_available) {
                $this->incrementRateLimitRedis($key, $expiry);
            } else {
                $this->incrementRateLimitFile($key, $expiry);
            }
        }
    }
    
    /**
     * Redis increment
     */
    private function incrementRateLimitRedis($key, $expiry) {
        try {
            $redis = new Redis();
            $redis->connect('127.0.0.1', 6379);
            
            $redis->multi();
            $redis->incr($key);
            $redis->expire($key, $expiry);
            $redis->exec();
            
        } catch (Exception $e) {
            error_log("Redis increment error: " . $e->getMessage());
        }
    }
    
    /**
     * File increment
     */
    private function incrementRateLimitFile($key, $expiry) {
        $file_path = sys_get_temp_dir() . "/rate_limit_$key";
        
        $data = ['count' => 1, 'timestamp' => time()];
        
        if (file_exists($file_path)) {
            $existing_data = json_decode(file_get_contents($file_path), true);
            
            if ($existing_data && time() - $existing_data['timestamp'] <= $expiry) {
                $data['count'] = $existing_data['count'] + 1;
                $data['timestamp'] = $existing_data['timestamp'];
            }
        }
        
        file_put_contents($file_path, json_encode($data), LOCK_EX);
    }
    
    /**
     * Check if user is blocked - FIXED VERSION
     */
    private function isUserBlocked($user_id) {
        global $db;
        
        try {
            // ✅ استفاده از getConnection() به جای دسترسی مستقیم
            $conn = $db->getConnection();
            if (!$conn) {
                return false;
            }
            
            $stmt = $conn->prepare("
                SELECT is_blocked, login_attempts 
                FROM users 
                WHERE user_id = ?
            ");
            $stmt->execute([$user_id]);
            $result = $stmt->fetch();
            
            if (!$result) {
                return false; // User doesn't exist, not blocked
            }
            
            // Check if explicitly blocked
            if ($result['is_blocked']) {
                return true;
            }
            
            // Check if too many failed login attempts
            if ($result['login_attempts'] >= $this->max_login_attempts) {
                return true;
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("User block check error: " . $e->getMessage());
            return false; // Fail open
        }
    }
    
    /**
     * Detect suspicious activity patterns - FIXED VERSION
     */
    private function isSuspiciousActivity($user_id) {
        global $db;
        
        try {
            // ✅ استفاده از getConnection() به جای دسترسی مستقیم
            $conn = $db->getConnection();
            if (!$conn) {
                return false;
            }
            
            // Check for too many requests in short time
            $stmt = $conn->prepare("
                SELECT COUNT(*) as count 
                FROM user_activities 
                WHERE user_id = ? 
                AND created_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE)
            ");
            $stmt->execute([$user_id]);
            $recent_activity = $stmt->fetch()['count'];
            
            if ($recent_activity > 50) { // More than 50 actions in 5 minutes
                return true;
            }
            
            // Check for repeated failed login attempts
            $stmt = $conn->prepare("
                SELECT COUNT(*) as count 
                FROM user_activities 
                WHERE user_id = ? 
                AND action LIKE '%failed%login%'
                AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
            ");
            $stmt->execute([$user_id]);
            $failed_logins = $stmt->fetch()['count'];
            
            if ($failed_logins > 10) {
                return true;
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("Suspicious activity check error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Validate username format
     */
    private function isValidUsername($username) {
        // Basic username validation
        if (empty($username)) {
            return true; // Username is optional
        }
        
        // Length check
        if (strlen($username) > 32) {
            return false;
        }
        
        // Character check (only alphanumeric and underscore)
        if (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
            return false;
        }
        
        // Check for suspicious patterns (removed captain/trade from suspicious list)
        $suspicious_patterns = [
            'bot', 'test', 'spam', 'fake'
    // admin و captain حذف شدند
    ];
        
        $username_lower = strtolower($username);
        foreach ($suspicious_patterns as $pattern) {
            if (strpos($username_lower, $pattern) !== false) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Log security events
     */
    private function logSecurityEvent($event_type, $identifier, $additional_info = null) {
        $log_entry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'event_type' => $event_type,
            'identifier' => $identifier,
            'ip' => $this->getClientIP(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            'additional_info' => $additional_info
        ];
        
        $log_message = json_encode($log_entry);
        error_log("SECURITY_EVENT: $log_message");
        
        // Also store in database for analysis
        $this->storeSecurityEvent($log_entry);
    }
    
    /**
     * Store security event in database - FIXED VERSION
     */
    private function storeSecurityEvent($event_data) {
        global $db;
        
        try {
            // ✅ استفاده از getConnection() به جای دسترسی مستقیم
            $conn = $db->getConnection();
            if (!$conn) {
                return false;
            }
            
            $stmt = $conn->prepare("
                INSERT INTO user_activities 
                (user_id, username, first_name, action, ip_address, user_agent, created_at)
                VALUES (?, ?, ?, ?, ?, ?, NOW())
            ");
            
            $stmt->execute([
                $event_data['identifier'],
                'security_log',
                'system',
                "SECURITY: {$event_data['event_type']} - {$event_data['additional_info']}",
                $event_data['ip'],
                $event_data['user_agent']
            ]);
            
        } catch (Exception $e) {
            error_log("Failed to store security event: " . $e->getMessage());
        }
    }
    
    /**
     * Sanitize input data
     */
    public function sanitizeInput($input, $type = 'text') {
        if (is_array($input)) {
            return array_map(function($item) use ($type) {
                return $this->sanitizeInput($item, $type);
            }, $input);
        }
        
        // Remove null bytes
        $input = str_replace("\0", '', $input);
        
        switch ($type) {
            case 'mobile':
                return preg_replace('/[^0-9]/', '', $input);
                
            case 'username':
                return preg_replace('/[^a-zA-Z0-9_]/', '', $input);
                
            case 'name':
                return preg_replace('/[<>"\']/', '', trim($input));
                
            case 'license':
                return preg_replace('/[^a-zA-Z0-9]/', '', $input);
                
            default:
                return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
        }
    }
    
    /**
     * Generate secure token
     */
    public function generateSecureToken($length = 32) {
        return bin2hex(random_bytes($length / 2));
    }
    
    /**
     * Hash sensitive data
     */
    public function hashSensitiveData($data, $salt = null) {
        if ($salt === null) {
            $salt = $this->generateSecureToken(16);
        }
        
        return [
            'hash' => hash('sha256', $data . $salt),
            'salt' => $salt
        ];
    }
    
    /**
     * Encrypt sensitive data
     */
    public function encryptData($data) {
        $key = hash('sha256', ENCRYPTION_KEY);
        $iv = random_bytes(16);
        
        $encrypted = openssl_encrypt($data, 'AES-256-CBC', $key, 0, $iv);
        
        return base64_encode($iv . $encrypted);
    }
    
    /**
     * Decrypt sensitive data
     */
    public function decryptData($encrypted_data) {
        $key = hash('sha256', ENCRYPTION_KEY);
        $data = base64_decode($encrypted_data);
        
        $iv = substr($data, 0, 16);
        $encrypted = substr($data, 16);
        
        return openssl_decrypt($encrypted, 'AES-256-CBC', $key, 0, $iv);
    }
    
    /**
     * Check for SQL injection patterns
     */
    public function detectSQLInjection($input) {
        $sql_patterns = [
            '/(\b(SELECT|INSERT|UPDATE|DELETE|DROP|CREATE|ALTER|EXEC|UNION)\b)/i',
            '/(\b(OR|AND)\s+\d+\s*=\s*\d+)/i',
            '/(\'|\")?\s*(OR|AND)\s*\1?\s*\d+\s*=\s*\d+/i',
            '/\b(UNION\s+SELECT|INFORMATION_SCHEMA)\b/i'
        ];
        
        foreach ($sql_patterns as $pattern) {
            if (preg_match($pattern, $input)) {
                $this->logSecurityEvent('sql_injection_attempt', 'unknown', $input);
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Check for XSS patterns
     */
    public function detectXSS($input) {
        $xss_patterns = [
            '/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi',
            '/javascript:/i',
            '/on\w+\s*=/i',
            '/<iframe/i',
            '/<object/i',
            '/<embed/i'
        ];
        
        foreach ($xss_patterns as $pattern) {
            if (preg_match($pattern, $input)) {
                $this->logSecurityEvent('xss_attempt', 'unknown', $input);
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get security status
     */
    public function getSecurityStatus() {
        return [
            'rate_limiting' => 'active',
            'user_validation' => 'active',
            'encryption' => 'active',
            'sql_injection_protection' => 'active',
            'xss_protection' => 'active',
            'redis_available' => $this->redis_available,
            'last_security_check' => date('Y-m-d H:i:s')
        ];
    }
}
?>